package hcrypto.engines;

import hcrypto.cipher.*;

/**
 * This class represents a key for a Vigenere cipher. The key
 * is entirely represented by a keyword String composed of characters from the chosen alphabet. 
 * <p>For this implementation of Vigenere cipher any of the alphabet ranges
 * can be used.
 * <P>See also:
 *  <BLOCKQUOTE> 
 *      <BR><a href="VigenereEngine.html">ViegenerEngine</a>
 *      <BR><a href="../cipher/Alphabet.html">Alphabet</a>
 *  </BLOCKQUOTE> 
 */

public class VigenereKey extends HistoricalKey {
  /**
   * A keyword description.
   */  
   public final static String DEFAULT_KEY_DESCRIPTOR_PROMPT_STRING = ("a keyword");
        
   
  /**
   * A default keyword.
   */ 
   public final static String DEFAULT_KEYWORD_STRING = ("keyword");
   
   /**
    * Initializes the VigenereKey with the specified keyspec.
    * @param keySpec takes the form "keyword/alphabet", i.e. 
    * "76TrombonesLEDTHEBIGPARADE/azAZ09" for example,
    * which would be mapped into the instance variables <i>keyword</i> 
    * as a String with the value of "76TrombonesLEDTHEBIGPARADE",
    * and <i>alphabet</i> as a String with the value "azAZ09".
    */ 
    public void init(String keyspec) throws Exception { 
        super.initKey(keyspec, false);  // false = don't remove keywords
        if (!isValidKeyword(keyword))
            throw new Exception("Vigenere keyword contains invalid characters for selected alphabet");
        this.blocksize = 1;
    }

   /**
    * Initializes the VigenereKey given a keyword and one or more alphabets.
    * @param keyword -- a string version of the keyword
    * @param alpha1, alpha2 -- Alphabets for plain/ciphertext respectively. May
    *  be identical.
    */ 
    public void init(String keyword, Alphabet alpha1, Alphabet alpha2) throws Exception { 
	if (alpha1 == null || alpha2 == null)
	    throw new Exception("VigenereKey.init(): Null alphabet reference passed");
	this.keyword = keyword;
        super.initKey(keyword, alpha1, alpha2);
        if (!isValidKeyword(keyword))
            throw new Exception("Vigenere keyword contains invalid characters for selected alphabet");
        this.blocksize = 1;
	//        System.out.println(keyword +  " " + blocksize);
    }



    /**
     *  This method overrides the superclass method. Vigenere doesn't
     *   really use the alphabets in its encryption engine. 
     */ 
    protected void initKeyArrays() throws Exception {
	if (alphabet.getSize() != cipherAlphabet.getSize()) 
	    throw new Exception ("ERROR: Plain and cipher alphabets must be the same size");

        char ch;  //Used in loops for storing a character

	plainKey = new char[alphabet.getSize()];
	if (cipherAlphabet != alphabet) 
	    cipherKey = new char[cipherAlphabet.getSize()];
	else 
	    cipherKey = new char[alphabet.getSize()];

        StringBuffer keychars = new StringBuffer("abcdefghijklmnopqrstuvwxyz");

	for (int k = 0; k < keychars.length(); k++) {
            ch = keychars.charAt(k);
//	    System.out.println("keychars.len = " + keychars.length() + " ch= " + ch);
            cipherKey[k] = ch;
            plainKey[cipherAlphabet.charToInt(ch)] = alphabet.intToChar(k);
        }//for
  } //initKeyArrays()}

   /**
    * Returns the name of the algorithm "Vigenere".
    */
      
    public String getAlgorithm() { 
        return "Vigenere"; 
    }   

    private boolean isValidKeyword(String s) {
        for (int k = 0; k < s.length(); k++)
            if (!alphabet.isInAlphabet(s.charAt(k)))
                return false;
        return true;
    }
}
