package hcrypto.provider;

import java.util.Hashtable;
import java.util.Vector;
import java.util.Enumeration;

/**
* This class represents a provider for the Historical Cipher API. A
* provider keeps a list of available cipher algorithms and their keys.
*
* Each provider has a name. The default provider, <A href="DefaultProvider.html">
* DefaultProvider</a>, is a subclass of this class that implements a number of
* example algorithms. To use the DefaultProvider in an application, use the following
* code:
* <pre>
*   public static void main (String args[]) throws Exception {
*       Provider.addProvider(new DefaultProvider("Default"));
*       Provider.addProvider(new MyProvider("Myname"));
*       ...
*   }
* </pre>
* In this case both the default provider and a user-defined provider
* are added to the list of providers that can be searched for implementations
* of cipher algorithms.
*
* <P>See also:
* <BLOCKQUOTE>
* <BR><A href="DefaultProvider.html">DefaultProvider</A>
* </BLOCKQUOTE>
*
* <P>To implement and use your own version of an algorithm, take the following
* steps:
* <UL>
*  <LI>Implement a subclass of <A href="../cipher/HistoricalKey.html">HistoricalKey</A>
*  <LI>Implement a subclass of <A  href="../cipher/BlockCipher.html">BlockCipher</A>
*  <LI>Implement a subclass of <A  href="Provider.html">Provider</A>
*  that places the name of your algorithm, the name of your BlockCipher subclass,
* and the name of your HistoricalKey subclass in the provider list.
* <LI>Place a call to <TT>Provider.add(Provider p)</TT> in <TT>main()</TT>  giving
* the name of your provider class as a reference.
* <LI>Create an instance of <A href="../cipher/Cipher.html">Cipher</A>
*  passing it the name  of your algorithm and your provider.
*  <LI>Create an instance of your <A href="../cipher/HistoricalKey.html">HistoricalKey</A>
* passing it the name of your algorithm and your provider.
*/

public class Provider {

/**
* The delimiter used to return a list of algorithm  names.
*/

public static final String DELIMITER = ",";

/**
* Stores the name of the provider.
*/

protected String name;

/**
* Stores references to the individual providers available at runtime.
*/

private static Vector providers  = new Vector();
protected Hashtable spiTable = new Hashtable();
protected Hashtable keyTable = new Hashtable();

/** * default constructor.
*/

public  Provider() {
}

public Provider(String s) {
  name = s;
}

/**
* Returns the name of this provider object.
*/

public String getName() {
  return name;
}

/**
*  Inserts the names of the engineClass and keyClass into the appropriate
*  tables thereby associating them with the named cipher.
* @param cipherName a   String giving the name of the cipher.
* @param engineClass a String giving the name of the associated engine.
* @param keyClass a String giving the name of the associated key.
*/

public void put(String cipherName, String engineClass,  String keyClass) {
  spiTable.put(cipherName, engineClass);
  keyTable.put(cipherName, keyClass);
}

/**
* Searches the main provider's  internal lists for an instance of
* a provider that implements the desired  algorithm.
* @param algorithm a String giving the name of the desired  algorithm.
*/

public String getCipherEngine (String algorithm) throws  Exception {
  int k = 0;
  boolean found = false;
  String spi = null;
  while (k < providers.size() && !found) {
    Provider p = (Provider) providers.elementAt(k);
       // System.out.println("Searching provider " + p.getName());
    if (p.spiTable.containsKey(algorithm)) {
    // System.out.println("Found algorithm " + algorithm);
      spi = (String) p.spiTable.get(algorithm);
      this.name = p.name;
      found = true;
    } else {
      k++;
    }// else
  } //while
  if (!found) throw new Exception("No such algorithm: " + algorithm);
  return spi;
}

  /**
  * Searches the main provider's internal lists for an instance of
  * a provider that implements the desired key.
  * @param algorithm a String giving the name of the desired algorithm.
  */

  public String getCipherKeyName (String algorithm) throws Exception {
    int k = 0;
    boolean found = false;
    String key =null;
    while (k < providers.size() && !found) {
      Provider p =(Provider) providers.elementAt(k);
      // System.out.println("Searching provider " + p.getName());
      if (p.keyTable.containsKey(algorithm)) {
        // System.out.println("Found algorithm " + algorithm);
        key = (String) p.keyTable.get(algorithm);
        this.name = p.name;
        found = true;
      } else {
        k++;
      }  //else
    } //while
    if (!found)
      throw new Exception("No such algorithm: " + algorithm);
    return key;
}

/**
* Searches the main provider's internal lists for an instance of
* the provider named in the second parameter that implements the desired algorithm.
* @param algorithm a String giving the name of the desired algorithm.
* @param provider a String giving the name of the desired provider.
*/

public String getCipherEngine (String algorithm, String provider) throws Exception {
  int k = 0;
  boolean found = false;
  String spi = null;
  while (k <providers.size() && !found) {
    Provider p = (Provider)
    providers.elementAt(k);
    if (p.getName().equals(provider)) {
      // System.out.println("Searching provider " + p.getName());
      if (p.spiTable.containsKey(algorithm)) {
        // System.out.println("Found algorithm " + algorithm + " in " + provider);
        spi = (String) p.spiTable.get(algorithm);
        this.name = p.name;
        found = true;
      } else throw new Exception("No engine for algorithm (" + algorithm + ") for provider " + provider);
    } else {
        k++;
    }
  } //while
  if (!found)
    throw new Exception("No such provider: " + provider);
  return spi;
}

  /**
  * Searches the main provider's internal lists for an instance of
  * the provider named in the second parameter that implements the desired algorithm.
  * @param algorithm a String giving the name of the desired algorithm.
  * @param provider a String giving the name of the desired provider.
  */

  public String getCipherKeyName (String algorithm, String provider) throws Exception {
    int k = 0;
    boolean found = false;
    String key = null;
    while (k < providers.size()&& !found) {
        Provider p = (Provider) providers.elementAt(k);
        if (p.getName().equals(provider)) {
              // System.out.println("Searching provider " + p.getName());
            if (p.keyTable.containsKey(algorithm)) {
              // System.out.println("Found algorithm " + algorithm + " in " + provider);
              key = (String) p.keyTable.get(algorithm);
              this.name = p.name;
              found = true;
            } else
              throw new Exception("No key for algorithm (" + algorithm + ") for provider " + provider);
        } else {
              k++;
        }
    }  //while
    if (!found)
      throw new Exception("No such provider: " + provider); return key;
  }

  /**
  * Adds the designated provider to the list of available providers.
  * @param p a reference to the provider object.
  */

  public static void addProvider(Provider p) {
    providers.addElement(p);
  }

  public static void addCipher(String cipher, String provider, String spiName) {
    Provider p = new Provider(provider);
    providers.addElement(p);
    // p.put(cipher, spiName);
    p.spiTable.put(cipher, spiName);
  }

  public static void addCipher(String cipher, String provider,String spiName, String keyName) {
    Provider p = new Provider(provider);
    providers.addElement(p);
        // p.put(cipher, spiName);
    p.spiTable.put(cipher, spiName);
    p.keyTable.put(cipher, keyName);
  }

  /**
  * Returns a comma-delimited string of the names
  * of the cipher algorithms. The names are returned in
  * the format "Algorithm(Provider)". For example "Caesar(Default)"
  * @param p a reference to the provider object.
  */

  public static String getCipherNames() {
    StringBuffer names = new StringBuffer();
    int j = 0;
    for ( int k = 0; k < providers.size(); k++) {
        Provider p = (Provider) providers.elementAt(k);
        // System.out.println("Size = " + p.size());
        // Enumeration enum = p.keys();
        Enumeration enum = p.spiTable.keys();
        while (enum.hasMoreElements()) {
          String algName = (String) enum.nextElement();
          // System.out.println("Name = " + algName);
          names.append(algName + " (" + p.getName() + ")" + DELIMITER);
          j++;
        } // while
    } //for
     return names.toString();
  }//getCipherNames()

  } //class Provider
